from flask import Flask, render_template, request, redirect, session, jsonify, url_for, flash
from flask_socketio import SocketIO
from flask_limiter import Limiter
from flask_limiter.util import get_remote_address
from flask_login import LoginManager, UserMixin, login_user, login_required, logout_user, current_user
from werkzeug.security import generate_password_hash, check_password_hash
from sqlalchemy import create_engine, Column, Integer, String, Float, DateTime
from sqlalchemy.orm import declarative_base, sessionmaker
from flask_caching import Cache
import subprocess, threading, psutil, time, os, re

from datetime import timedelta, datetime
import json
import secrets

BASE_DIR = r"C:\path\to\your\project"
TEMPLATES_DIR = os.path.join(BASE_DIR, "templates")
STATIC_DIR = os.path.join(BASE_DIR, "static")

BAT_FILE = r"C:\path\to\your\minecraft\server.bat"
SERVER_DIR = r"C:\path\to\your\minecraft\server"

app = Flask(
    __name__,
    template_folder=TEMPLATES_DIR,
    static_folder=STATIC_DIR,
    static_url_path='/static'
)
socketio = SocketIO(app, cors_allowed_origins="*")
app.secret_key = "your_secret_key_here"
app.permanent_session_lifetime = timedelta(hours=1)

# Initialize caching
cache = Cache(app, config={'CACHE_TYPE': 'simple'})

# CSRF Protection
@app.before_request
def csrf_protect():
    if request.method == "POST":
        token = session.get('_csrf_token')
        if not token:
            return jsonify({"error": "CSRF token missing or incorrect"}), 403
        # Check for form data (traditional forms)
        if request.form.get('_csrf_token') != token:
            # Check for header (AJAX requests)
            if request.headers.get('X-CSRFToken') != token:
                return jsonify({"error": "CSRF token missing or incorrect"}), 403

def generate_csrf_token():
    if '_csrf_token' not in session:
        session['_csrf_token'] = secrets.token_hex(16)
    return session['_csrf_token']

# Database setup
engine = create_engine('sqlite:///webconsole.db', echo=False)
Base = declarative_base()

class LogEntry(Base):
    __tablename__ = 'logs'
    id = Column(Integer, primary_key=True)
    timestamp = Column(DateTime, default=datetime.utcnow)
    line = Column(String)

class StatEntry(Base):
    __tablename__ = 'stats'
    id = Column(Integer, primary_key=True)
    timestamp = Column(DateTime, default=datetime.utcnow)
    cpu = Column(Float)
    ram = Column(Float)
    temp = Column(Float)
    players = Column(Integer)
    tps = Column(Float)

Base.metadata.create_all(engine)
Session = sessionmaker(bind=engine)

# Rate Limiting
limiter = Limiter(
    app=app,
    key_func=get_remote_address,
    default_limits=["100 per minute"]
)

# Specific rate limits for sensitive endpoints
kick_ban_limiter = limiter.limit("10 per minute")
config_limiter = limiter.limit("5 per minute")

# Login Manager
login_manager = LoginManager()
login_manager.init_app(app)
login_manager.login_view = 'login'

# User Model
class User(UserMixin):
    def __init__(self, id, username, password_hash):
        self.id = id
        self.username = username
        self.password_hash = password_hash

# Simple user storage (in production, use a database)
users = {
    "admin": User(1, "admin", generate_password_hash("example_password"))
}

@login_manager.user_loader
def load_user(user_id):
    for user in users.values():
        if user.id == int(user_id):
            return user
    return None

console_output = []
server_process = None
players = 0
player_list = []
tps = 20
server_start_time = None
last_player_update = 0

if not os.path.exists(STATIC_DIR):
    print(f"Warning: Static folder not found at {STATIC_DIR}. Creating it.")
    os.makedirs(STATIC_DIR, exist_ok=True)
if not os.path.exists(TEMPLATES_DIR):
    print(f"Warning: Templates folder not found at {TEMPLATES_DIR}. Creating it.")
    os.makedirs(TEMPLATES_DIR, exist_ok=True)
if not os.path.exists(BAT_FILE):
    print(f"Warning: Server batch file not found at {BAT_FILE}. Please update the path in the script.")

def read_console():
    global server_process, console_output, players, tps, player_list, last_player_update
    if server_process is None: return
    db_session = Session()
    try:
        for line in iter(server_process.stdout.readline, b''):
            line_str = line.decode(errors='ignore').strip()
            console_output.append(line_str)
            if len(console_output) > 100: console_output = console_output[-100:]
            # Save to DB asynchronously
            log_entry = LogEntry(line=line_str)
            db_session.add(log_entry)
            db_session.commit()
            # Parse players
            if "joined" in line_str.lower():
                players += 1
                # Extract player name from log line
                match = re.search(r'(\w+)\s+joined', line_str, re.IGNORECASE)
                if match:
                    player_name = match.group(1)
                    if player_name not in player_list:
                        player_list.append(player_name)
            elif "left" in line_str.lower():
                players = max(0, players - 1)
                # Extract player name from log line
                match = re.search(r'(\w+)\s+left', line_str, re.IGNORECASE)
                if match:
                    player_name = match.group(1)
                    if player_name in player_list:
                        player_list.remove(player_name)
            # Parse TPS (example: "TPS: 19.5")
            if "TPS:" in line_str:
                try:
                    tps_str = line_str.split("TPS:")[1].strip().split()[0]
                    tps = float(tps_str)
                except: pass
            # Parse list command output
            if "players online:" in line_str:
                match = re.search(r'players online:\s*(.+)', line_str)
                if match:
                    players_str = match.group(1)
                    current_players = [p.strip() for p in players_str.split(',') if p.strip()]
                    player_list = current_players
                    players = len(player_list)
            # Update player list periodically (every 30 seconds) or when players change
            current_time = time.time()
            if current_time - last_player_update > 30 or "joined the game" in line_str or "left the game" in line_str:
                last_player_update = current_time
                # Send list command to get current players
                if server_process and server_process.poll() is None:
                    try:
                        server_process.stdin.write(b"list\n")
                        server_process.stdin.flush()
                    except:
                        pass
            socketio.emit('console_update', {'line': line_str})
    finally:
        db_session.close()

def start_server():
    def run_server():
        global server_process
        if server_process is None or server_process.poll() is not None:
            server_process = subprocess.Popen(
                BAT_FILE,
                cwd=SERVER_DIR,
                stdout=subprocess.PIPE,
                stderr=subprocess.STDOUT,
                stdin=subprocess.PIPE,
                bufsize=1,
                shell=True
            )
            threading.Thread(target=read_console, daemon=True).start()
    threading.Thread(target=run_server, daemon=True).start()

def emit_status():
    db_session = Session()
    try:
        while True:
            cpu = int(psutil.cpu_percent())
            ram = int(psutil.virtual_memory().percent)
            temp = None
            try:
                temps = psutil.sensors_temperatures()
                if temps: temp = int(list(temps.values())[0][0].current)
            except: pass
            global players, tps
            server_status = "Running" if server_process and server_process.poll() is None else "Stopped"
            # Save stats to DB
            stat_entry = StatEntry(cpu=cpu, ram=ram, temp=temp, players=players, tps=tps)
            db_session.add(stat_entry)
            db_session.commit()
            socketio.emit('status_update', {
                "cpu": cpu,
                "ram": ram,
                "temp": temp,
                "players": players,
                "tps": tps,
                "server_status": server_status
            })
            socketio.sleep(5)  # Reduced polling to 5 seconds
    finally:
        db_session.close()

threading.Thread(target=emit_status, daemon=True).start()

@socketio.on('connect')
def handle_connect():
    for line in console_output:
        socketio.emit('console_update', {'line': line})

def stop_server():
    global server_process, server_start_time
    if server_process and server_process.poll() is None:
        try:
            server_process.stdin.write(b"stop\n")
            server_process.stdin.flush()
            time.sleep(5)  # Wait for graceful shutdown
            if server_process.poll() is None:
                server_process.terminate()
        except:
            server_process.terminate()
        server_process = None
        server_start_time = None

@app.route("/", methods=["GET"])
def index_redirect():
    if current_user.is_authenticated: return redirect("/dashboard")
    return redirect("/login")

@app.route("/login", methods=["GET", "POST"])
@limiter.limit("5 per minute")
def login():
    if current_user.is_authenticated:
        return redirect(url_for('dashboard'))
    error = None
    if request.method == "POST":
        username = request.form.get("username")
        password = request.form.get("password")
        user = users.get(username)
        if user and check_password_hash(user.password_hash, password):
            login_user(user)
            return redirect(url_for('dashboard'))
        else:
            error = "Invalid username or password"
            flash(error)
    return render_template("login.html", error=error, csrf_token=generate_csrf_token())

@app.route("/logout")
@login_required
def logout():
    logout_user()
    return redirect(url_for('login'))

@app.route("/dashboard", methods=["GET"])
@login_required
def dashboard():
    cpu = int(psutil.cpu_percent())
    ram = int(psutil.virtual_memory().percent)
    temp = None
    try:
        temps = psutil.sensors_temperatures()
        if temps: temp = int(list(temps.values())[0][0].current)
    except: pass
    global players, tps
    return render_template("index.html", cpu=cpu, ram=ram, temp=temp, players=players, tps=tps, csrf_token=generate_csrf_token())

@app.route("/status", methods=["GET"])
@login_required
@cache.cached(timeout=10)  # Cache for 10 seconds
def status():
    cpu = int(psutil.cpu_percent())
    ram = int(psutil.virtual_memory().percent)
    temp = None
    try:
        temps = psutil.sensors_temperatures()
        if temps: temp = int(list(temps.values())[0][0].current)
    except: pass
    global players, tps, server_start_time
    server_status = "Running" if server_process and server_process.poll() is None else "Stopped"
    uptime = None
    if server_start_time and server_status == "Running":
        uptime = str(datetime.utcnow() - server_start_time).split('.')[0]  # Format as HH:MM:SS
    return jsonify({
        "cpu": cpu,
        "ram": ram,
        "temp": temp,
        "players": players,
        "tps": tps,
        "server_status": server_status,
        "uptime": uptime,
        "console": console_output
    })

@app.route("/command", methods=["POST"])
@login_required
def command():
    cmd = request.form.get("cmd")
    if server_process and server_process.poll() is None and cmd:
        server_process.stdin.write((cmd + "\n").encode())
        server_process.stdin.flush()
        return jsonify({"status": "success"})
    return jsonify({"status": "error", "message": "Server not running or invalid command"})

@app.route("/control/<action>", methods=["POST"])
@login_required
def control(action):
    if action == "start": start_server()
    elif action == "stop": stop_server()
    elif action == "restart":
        stop_server()
        time.sleep(2)
        start_server()
    return jsonify({"status": "success", "action": action})

@app.route("/players", methods=["GET"])
@login_required
def get_players():
    global player_list
    return jsonify({"players": player_list})

@app.route("/kick/<player>", methods=["POST"])
@login_required
@kick_ban_limiter
def kick_player(player):
    if server_process and server_process.poll() is None:
        server_process.stdin.write(f"kick {player}\n".encode())
        server_process.stdin.flush()
        # Update player list after kick
        global player_list
        if player in player_list:
            player_list.remove(player)
        return jsonify({"status": "success"})
    return jsonify({"status": "error", "message": "Server not running"})

@app.route("/ban/<player>", methods=["POST"])
@login_required
@kick_ban_limiter
def ban_player(player):
    if server_process and server_process.poll() is None:
        server_process.stdin.write(f"ban {player}\n".encode())
        server_process.stdin.flush()
        # Update player list after ban
        global player_list
        if player in player_list:
            player_list.remove(player)
        return jsonify({"status": "success"})
    return jsonify({"status": "error", "message": "Server not running"})

@app.route("/config", methods=["GET"])
@login_required
def get_config():
    config_path = os.path.join(SERVER_DIR, "server.properties")
    if os.path.exists(config_path):
        with open(config_path, 'r') as f:
            config = f.read()
        return jsonify({"config": config})
    else:
        return jsonify({"error": "Config file not found"}), 404

@app.route("/config", methods=["POST"])
@login_required
@config_limiter
def save_config():
    data = request.get_json()
    config = data.get("config")
    config_path = os.path.join(SERVER_DIR, "server.properties")
    try:
        with open(config_path, 'w') as f:
            f.write(config)
        return jsonify({"status": "success"})
    except Exception as e:
        return jsonify({"status": "error", "message": str(e)}), 500

if __name__ == "__main__":
    try:
        socketio.run(app, host="0.0.0.0", port=8080, debug=False)
    except OSError as e:
        if "10048" in str(e) or "address already in use" in str(e).lower():
            print("Port 8080 is already in use. Please free up port 8080 or update your TCP tunnel configuration to match a different port.")
            print("To find what's using port 8080, run: netstat -ano | findstr :8080")
            raise
        else:
            raise
