#!/bin/bash
# Minecraft Server Startup Script

# Server configuration
SERVER_JAR="server.jar"
MIN_RAM="1024M"
MAX_RAM="2048M"
JAVA_ARGS="-Xmx$MAX_RAM -Xms$MIN_RAM -XX:+UseG1GC -XX:+ParallelRefProcEnabled -XX:MaxGCPauseMillis=200 -XX:+UnlockExperimentalVMOptions -XX:+DisableExplicitGC -XX:+AlwaysPreTouch -XX:G1NewSizePercent=30 -XX:G1MaxNewSizePercent=40 -XX:G1HeapRegionSize=8M -XX:G1ReservePercent=20 -XX:G1HeapWastePercent=5 -XX:G1MixedGCCountTarget=4 -XX:InitiatingHeapOccupancyPercent=15 -XX:G1MixedGCLiveThresholdPercent=90 -XX:G1RSetUpdatingPauseTimePercent=5 -XX:SurvivorRatio=32 -XX:+PerfDisableSharedMem -XX:MaxTenuringThreshold=1 -Dusing.aikars.flags=https://mcflags.emc.gs -Daikars.new.flags=true"

# Function to display usage
usage() {
    echo "Usage: $0 {start|stop|restart|status}"
    exit 1
}

# Function to check if server is running
is_running() {
    if pgrep -f "$SERVER_JAR" > /dev/null; then
        return 0
    else
        return 1
    fi
}

# Function to start server
start_server() {
    if is_running; then
        echo "Server is already running!"
        exit 1
    fi

    echo "Starting Minecraft server..."
    cd "$(dirname "$0")"
    screen -dmS minecraft java $JAVA_ARGS -jar $SERVER_JAR nogui

    # Wait a moment for server to start
    sleep 5

    if is_running; then
        echo "Server started successfully!"
    else
        echo "Failed to start server!"
        exit 1
    fi
}

# Function to stop server
stop_server() {
    if ! is_running; then
        echo "Server is not running!"
        exit 1
    fi

    echo "Stopping Minecraft server..."
    screen -S minecraft -p 0 -X stuff "stop$(printf '\r')"

    # Wait for server to stop
    for i in {1..30}; do
        if ! is_running; then
            echo "Server stopped successfully!"
            exit 0
        fi
        sleep 1
    done

    echo "Server did not stop gracefully, forcing shutdown..."
    pkill -f "$SERVER_JAR"
    echo "Server forcefully stopped!"
}

# Function to restart server
restart_server() {
    echo "Restarting Minecraft server..."
    stop_server
    sleep 5
    start_server
}

# Function to check server status
server_status() {
    if is_running; then
        echo "Server is running"
        # Get player count if possible
        screen -S minecraft -p 0 -X stuff "list$(printf '\r')"
    else
        echo "Server is not running"
    fi
}

# Main script logic
case "$1" in
    start)
        start_server
        ;;
    stop)
        stop_server
        ;;
    restart)
        restart_server
        ;;
    status)
        server_status
        ;;
    *)
        usage
        ;;
esac
